function [] = BattProfile(batt, prof, opt)
arguments
    batt struct
    prof struct
    opt.cellLevel = false
    opt.simulations struct
    opt.FontSize = 16,
    opt.FontName = 'Arial';
end
%Battprofile
% Plot the battery power, energy, current and Ah throughput profiles
%
% Input arguments
% ---------------
% batt : struct
%   battery data structure.
% prof : struct
%   Time profiles data structure.
% Outputs
% ---------------
% fig : Figure
%   Figure handle of the plot.
% ax : Axes
%   Axes handle of the plot.


if isfield(opt,'simulations')
    n = opt.simulations.number_of_simulations;
    array = opt.simulations.array;
    legend_ = opt.simulations.legend;
    if n>0
        cmap = [linspace(.9,0,n)', linspace(.9447,.447,n)', linspace(.9741,.741,n)'];
    end
end

% Plot battery profiles
% Compute quantities to plot
time = [prof.time];
distance_travelled = cumtrapz(time,[prof.vehSpd]); %m
battCurr = [prof.battCurr];
% rmsCurr = rms(battCurr)
battPower = [prof.battCurr].*[prof.battVolt];
% rmsPower = rms(battPower)
if opt.cellLevel
    battCurr = battCurr./batt.numParallel;
    battPower = battPower./(batt.numParallel.*batt.numSeries);
    cellLevelLabel = 'cell';
else
    cellLevelLabel = 'pack';
end
Ah_throughput = cumtrapz(time,abs(battCurr))/3600;
battEnergy = cumtrapz(time, battPower);

fig_num = 100 + opt.cellLevel;
if ~ishandle(fig_num)
    figure(fig_num)
    t = tiledlayout(2,2);
    t.TileSpacing = 'compact';
    t.Padding = "tight";
    txt = xlabel(t,'Distance travelled (thousands of kms)');
    txt.FontSize = opt.FontSize;
    txt.FontName = opt.FontName;
end
figure(fig_num)

% fig.Units = "centimeters";
% fig.Position = [0 0 19 24];
% % per TiledLayout
% t.TileSpacing = "compact";
% t.Padding = "tight";

nexttile(1); hold on; box on; grid on;
plot(distance_travelled/1000, battCurr,  'LineWidth', 1.5)
ylabel(append("I_{",cellLevelLabel,"} (A)"))

nexttile(2); hold on; box on; grid on;
plot(distance_travelled/1000, Ah_throughput, 'LineWidth', 1.5)
ylabel(append("Ah_{",cellLevelLabel,"} (Ah)"))

nexttile(3); hold on; box on; grid on;
plot(distance_travelled/1000, battPower/1000, 'LineWidth', 1.5)
ylabel(append("P_{",cellLevelLabel,"} (kW)"))

nexttile(4); hold on; box on; grid on;
if opt.cellLevel
    plot(distance_travelled/1000, battEnergy/10^3, 'LineWidth', 1.5)
    ylabel(append("E_{",cellLevelLabel,"} (kJ)"))
else
    plot(distance_travelled/1000, battEnergy/10^6, 'LineWidth', 1.5)
    ylabel(append("E_{",cellLevelLabel,"} (MJ)"))
end

if isfield(opt,'simulations'); colororder(cmap); end


set(findall(gcf,'-property','FontSize'),'FontSize',opt.FontSize)
set(findall(gcf,'-property','FontName'),'FontName',opt.FontName)



end